var express = require('express');
var router = express.Router();

const authenticateHeader = require('../middleware/middleware').authenticateHeader;
const authenticateJWT = require('../middleware/middleware').authenticateJWT;

const nodecron = require('../controllers/nodeScheduleController');
const UserController = require('../controllers/UserController');
const AuthController = require('../controllers/AuthController');
const IndexController = require('../controllers/IndexController');
const PageController = require('../controllers/PageController');
const VehicleController = require('../controllers/VehicleController');
const BankController = require('../controllers/BankController');
const BookingController = require('../controllers/BookingController');
const RatingController = require('../controllers/RatingController');
const CouponController = require('../controllers/CouponController');
const PaymentController = require('../controllers/PaymentController');
const WithdrawalController = require('../controllers/WithdrawalController');
const NotificationController = require('../controllers/NotificationController');
const TipController = require('../controllers/TipController');

module.exports = (io) => {
    nodecron(io);
    router.post("/check-email", authenticateHeader, AuthController.checkEmail);
    router.post("/sign-in", authenticateHeader, AuthController.signIn);

    // User
    router.post("/user-signup", authenticateHeader, UserController.userSignUp);
    router.post("/user-profile-update", authenticateHeader, authenticateJWT, UserController.userProfileUpdate);
    router.post("/user-profile", authenticateHeader, authenticateJWT, UserController.userProfile);

    router.post("/update-current-details", authenticateHeader, authenticateJWT, UserController.updateCurrentDetails);

    // Email otp
    router.post("/send-email-otp", authenticateHeader, IndexController.sendEmailOtp);
    router.post("/verify-otp", authenticateHeader, IndexController.verifyOtp);

    // Phone otp
    router.post("/send-phone-otp", authenticateHeader, IndexController.sendPhoneOtp);
    router.post("/verify-phone-otp", authenticateHeader, IndexController.verifyPhoneOtp);

    // Forgot Password
    router.post("/forgot-password", authenticateHeader, IndexController.forgotPassword);
    router.get("/reset-password/:id", IndexController.resetPassword);
    router.post("/updateResetPassword", IndexController.updatePassword);

    router.post("/change-password", authenticateHeader, authenticateJWT, IndexController.changePassword);
    router.post("/set-new-password", authenticateHeader, IndexController.setNewPassword);


    // Cms
    router.post("/cms", authenticateHeader, PageController.cms);

    // Faq
    router.post("/faq", authenticateHeader, PageController.faq);

    // Check push
    router.post("/check-push", authenticateHeader, IndexController.checkPush);

    // Logout
    router.post("/logout", authenticateHeader, authenticateJWT, IndexController.logout);

    // Notification On/Off
    router.post("/update-notification-status", authenticateHeader, authenticateJWT, IndexController.updateNotificationStatus);

    // Account delete
    router.post("/delete-account", authenticateHeader, authenticateJWT, IndexController.deleteAccount);

    // Contact us
    router.post("/contact-us", authenticateHeader, authenticateJWT, PageController.contactUs);


    // VehicleController
    router.post("/vehicle-type", authenticateHeader, VehicleController.vehicleType);

    router.post("/vehicle-update", authenticateHeader, authenticateJWT, VehicleController.vehicleUpdate);

    router.post("/vehicle-model", authenticateHeader, VehicleController.vehicleModel);

    // Bank
    router.post("/add-bank-account", authenticateHeader, authenticateJWT, BankController.addBankAccount);
    router.post("/bank-account-list", authenticateHeader, authenticateJWT, BankController.bankAccountList);
    router.post("/bank-delete", authenticateHeader, authenticateJWT, BankController.bankDelete);
    router.post("/set-default-bank", authenticateHeader, authenticateJWT, BankController.setDefaultBank);

    // Booking
    router.post("/create-booking", authenticateHeader, authenticateJWT, BookingController.createBooking(io));
    router.post("/booking-details", authenticateHeader, authenticateJWT, BookingController.bookingDetails);
    router.post("/send-booking-request", authenticateHeader, authenticateJWT, BookingController.sendBookingRequest(io));
    router.post("/driver-booking-request", authenticateHeader, authenticateJWT, BookingController.driverBookingRequest);
    router.post("/my-bookings", authenticateHeader, authenticateJWT, BookingController.myBookings);
    router.post("/driver-bookings", authenticateHeader, authenticateJWT, BookingController.driverBookings);

    // Booking cancel
    router.post("/booking-cancel", authenticateHeader, authenticateJWT, BookingController.bookingCancel(io));

    // driver bookinglist
    router.post("/driver-new-bookings-list", authenticateHeader, authenticateJWT, BookingController.driverNewBookingList);

    router.post("/delete-expired-bookings", authenticateHeader, BookingController.deleteExpiredBookings);

    // Driver
    router.post("/driver-accept-reject-booking", authenticateHeader, authenticateJWT, BookingController.driverAcceptRejectBooking(io));

    // User
    router.post("/user-accept-reject-driver", authenticateHeader, authenticateJWT, BookingController.userAcceptRejectDriver(io));

    // Rating
    router.post("/add-rating", authenticateHeader, authenticateJWT, RatingController.addRating(io));
    router.post("/driver-rating", authenticateHeader, RatingController.driverRating);


    // CouponController
    router.get("/coupon-list", authenticateHeader, authenticateJWT, CouponController.couponList);
    router.post("/coupon-apply", authenticateHeader, authenticateJWT, CouponController.couponApply);

    // Update medical details
    router.post("/medical-details-update", authenticateHeader, authenticateJWT, UserController.medicalDetailsUpdate);

    // Price
    router.get("/price", authenticateHeader, UserController.price);

    // Payment
    router.post("/payment", authenticateHeader, authenticateJWT, PaymentController.payment);
    router.post("/stripe-webhook", PaymentController.stripeWebhook);
    router.post("/fetch-payment-status", PaymentController.fetchPaymentStatus);

    // Withdrawal
    router.post("/withdrawal-request", authenticateHeader, authenticateJWT, WithdrawalController.withdrawalRequest);
    router.post("/all-withdrawal-requests", authenticateHeader, authenticateJWT, WithdrawalController.allWithdrawalRequests);
    router.post("/transfer-money", authenticateHeader, authenticateJWT, WithdrawalController.tranferMoney);

    router.post("/stripeConnect", authenticateHeader, authenticateJWT, PaymentController.stripeConnect);
    router.get("/stripe_connect_retrun", PaymentController.stripe_connect_retrun)

    // Notification
    router.post("/notification-list", authenticateHeader, authenticateJWT, NotificationController.notificationList);
    router.post("/clear-notification", authenticateHeader, authenticateJWT, NotificationController.clearNotification);
    router.post("/delete-notification", authenticateHeader, authenticateJWT, NotificationController.deleteNotification);

    // Tip
    router.post("/tip", authenticateHeader, authenticateJWT, TipController.tip);
    router.post("/tip-stripe-webhook", TipController.tipStripeWebhook);

    return router;
}
// module.exports = router;