"use strict";
/**
 * Copyright 2024 Google LLC. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnabledTraceUtil = void 0;
const api_1 = require("@opentelemetry/api");
const span_1 = require("./span");
class EnabledTraceUtil {
    constructor(settings) {
        var _a;
        let tracerProvider = (_a = settings.openTelemetryOptions) === null || _a === void 0 ? void 0 : _a.tracerProvider;
        // If a TracerProvider has not been given to us, we try to use the global one.
        if (!tracerProvider) {
            const { trace } = require('@opentelemetry/api');
            tracerProvider = trace.getTracerProvider();
        }
        const libVersion = require('../../../package.json').version;
        const libName = require('../../../package.json').name;
        this.tracer = tracerProvider.getTracer(libName, libVersion);
    }
    endSpan(otelSpan, error) {
        otelSpan.setStatus({
            code: api_1.SpanStatusCode.ERROR,
            message: error.message,
        });
        otelSpan.recordException(error);
        otelSpan.end();
    }
    startActiveSpan(name, fn, attributes) {
        return this.tracer.startActiveSpan(name, {
            attributes: attributes,
        }, (otelSpan) => {
            // Note that if `fn` returns a `Promise`, we want the otelSpan to end
            // after the `Promise` has resolved, NOT after the `fn` has returned.
            // Therefore, we should not use a `finally` clause to end the otelSpan.
            try {
                let result = fn(new span_1.Span(otelSpan));
                if (result instanceof Promise) {
                    result = result
                        .then(value => {
                        otelSpan.end();
                        return value;
                    })
                        .catch(error => {
                        this.endSpan(otelSpan, error);
                        // Returns a Promise.reject the same as the underlying function.
                        return Promise.reject(error);
                    });
                }
                else {
                    otelSpan.end();
                }
                return result;
            }
            catch (error) {
                this.endSpan(otelSpan, error);
                // Re-throw the exception to maintain normal error handling.
                throw error;
            }
        });
    }
    startSpan(name) {
        return new span_1.Span(this.tracer.startSpan(name, undefined, api_1.context.active()));
    }
    currentSpan() {
        return new span_1.Span(api_1.trace.getActiveSpan());
    }
}
exports.EnabledTraceUtil = EnabledTraceUtil;
//# sourceMappingURL=enabled-trace-util.js.map