const models = require('../models');
const helper = require('../helpers/helper')
const { Validator } = require('node-input-validator');
const Sequelize = require("sequelize");
const { Op, fn, col } = Sequelize;
const cron = require('node-cron');
const stripe = require('stripe')('sk_test_51PvYN9AZ3KtiMZbRRnumfMRxTvGl3vq20RmJyEsj7HlLaOZJD0uYJaq94bQX2cWLWgbqEymTEYpTtPPyw1QxUp2M002EQfOIyC');
module.exports = (io) => {
    // send request to drivers every 30 seconds untill driver not accept
    cron.schedule("*/30 * * * * *", async () => {
        try {
            let bookings = await models.bookings.findAll({
                include: [
                    {
                        model: models.vehicle_types,
                    }
                ],
                where: {
                    driver_id: 0,
                    status: 0
                }
            });

            const TEN_MINUTES = 10 * 60 * 1000;

            if (bookings && bookings.length > 0) {
                for (let booking of bookings) {

                    const currentDate = new Date();
                    const bookingTimeString = booking.booking_time; // Assuming format 'HH:mm'
                    const [hours, minutes] = bookingTimeString.split(':').map(Number);

                    const bookingTime = new Date(currentDate);
                    bookingTime.setHours(hours);
                    bookingTime.setMinutes(minutes);
                    bookingTime.setSeconds(0);
                    bookingTime.setMilliseconds(0);

                    const timeDifference = currentDate - bookingTime;

                    if (timeDifference > TEN_MINUTES) {
                        console.log(`Cancelling booking ID ${booking.id} due to timeout.`);
                        // Refund process (if stripe_status is 'succeeded')
                        if (booking.stripe_status === 'succeeded' && booking.transaction_id) {
                            try {
                                const amountInCents = Math.round(booking.price * 100);
                                const refund = await stripe.refunds.create({
                                    payment_intent: booking.transaction_id,
                                    amount: amountInCents
                                });

                                console.log(`Refund successful for booking ID ${booking.id}:`, refund);

                                // Update booking with refund data and mark as cancelled (status 6)
                                await models.bookings.update({
                                    cancel_data: refund,
                                    cancel_amount: booking.price,
                                    status: 6 // Mark booking as expired
                                }, {
                                    where: { id: booking.id }
                                });
                            } catch (refundError) {
                                console.error(`Error processing refund for booking ID ${booking.id}:`, refundError);
                            }
                        } else {
                            // If no Stripe payment, simply mark the booking as cancelled
                            await models.bookings.update({
                                status: 6 // Mark booking as cancelled
                            }, {
                                where: { id: booking.id }
                            });
                        }

                        // Delete related driver requests
                        await models.request_send_to_drivers.destroy({
                            where: { booking_id: booking.id }
                        });
                        console.log(`Booking ID ${booking.id} cancelled and related driver requests deleted.`);
                        // Skip the rest of the loop for this booking, as it's been cancelled
                        continue;
                    }

                    if (booking.stripe_status == 'succeeded') {
                        const whereCondition = {
                            role: 1,
                            vehicle_type: booking.vehicle_id,
                            id: {
                                [Op.ne]: booking.user_id,
                            },
                        };

                        let potentialDrivers = await models.users.findAll({
                            attributes: [
                                "id",
                                "first_name",
                                "last_name",
                                "email",
                                "country_code",
                                "phone_number",
                                "latitude",
                                "longitude",
                                "device_type",
                                "device_token",
                                [
                                    Sequelize.literal(
                                        `6371 * acos(cos(radians(${booking.destination_from_lat})) * cos(radians(latitude)) * cos(radians(${booking.destination_from_long}) - radians(longitude)) + sin(radians(${booking.destination_from_lat})) * sin(radians(latitude)))`
                                    ),
                                    "distance",
                                ],
                            ],
                            where: whereCondition,
                            having: {
                                distance: {
                                    [Op.lte]: 500, // distance in kilometers
                                },
                            },
                            raw: true,
                            order: [["id", "desc"]],
                        });

                        if (potentialDrivers && potentialDrivers.length > 0) {
                            for (let driver of potentialDrivers) {
                                // Check if a request has already been sent to this driver for this booking
                                let existingRequest = await models.request_send_to_drivers.findOne({
                                    where: {
                                        booking_id: booking.id,
                                        driver_id: driver.id,
                                    },
                                    raw: true,
                                });

                                if (!existingRequest) {
                                    // If no existing request, create a new one
                                    await models.request_send_to_drivers.create({
                                        booking_id: booking.id,
                                        driver_id: driver.id,
                                    });

                                    let message = `New booking from ${booking.destination_from} to ${booking.destination_to}`;
                                    let type = 1;
                                    let data = {
                                        sender_id: booking.user_id,
                                        receiver_id: driver.id,
                                        user_type: driver.role,
                                        message: message,
                                        type: type,
                                        booking_id: booking.id,
                                        device_token: driver.device_token,
                                        device_type: driver.device_type,
                                    };
                                    console.log("Sending push notification data:", data);
                                    // Uncomment and implement the following line to send push notifications
                                    // helper.sendPushToAndroid(data);

                                    let driverSocket = await models.socket_users.findOne({
                                        where: {
                                            user_id: driver.id,
                                        },
                                        raw: true,
                                    });

                                    if (driverSocket && driverSocket.socket_id) {
                                        io.to(driverSocket.socket_id).emit('bookingRequest', booking);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (error) {
            console.log("Error occurred during booking request cron job:", error);
        }
    });

    // Cron job scheduled to run at midnight (12:00 AM)
    cron.schedule("0 0 * * *", async () => {
        try {
            let expiredBookings = await models.bookings.findAll({
                where: {
                    booking_date: {
                        [Op.lt]: new Date() // Find bookings where booking_date is earlier than today
                    },
                    status: 0
                },
                attributes: ['id', 'transaction_id', 'stripe_status', 'cancel_amount']
            });

            if (expiredBookings.length > 0) {

                for (const booking of expiredBookings) {
                    const bookingId = booking.id;

                    // If the booking has a successful Stripe payment, initiate refund
                    if (booking.transaction_id && booking.stripe_status === 'succeeded') {
                        try {
                            const amountInCents = Math.round(booking.price * 100);

                            // Create refund
                            const refund = await stripe.refunds.create({
                                payment_intent: booking.transaction_id,
                                amount: amountInCents,
                            });

                            console.log(`Refund successful for booking ID ${bookingId}:`, refund);

                            // Update booking with refund data and set status to "expired" (6)
                            await models.bookings.update({
                                cancel_data: refund,
                                cancel_amount: booking.price,
                                status: 6 // Mark booking as expired
                            }, {
                                where: { id: bookingId }
                            });

                        } catch (refundError) {
                            console.error(`Error processing refund for booking ID ${bookingId}:`, refundError);
                        }
                    } else {
                        // If no transaction or stripe_status is not 'succeeded', just mark as expired
                        await models.bookings.update({
                            status: 6 // Mark booking as expired
                        }, {
                            where: { id: bookingId }
                        });
                    }

                    // Delete related records from request_send_to_drivers table
                    await models.request_send_to_drivers.destroy({
                        where: {
                            booking_id: bookingId
                        }
                    });

                    console.log(`Booking ID ${bookingId} marked as expired and related records cleaned up.`);
                }
            } else {
                console.log('No expired bookings found.');
            }
        } catch (error) {
            console.log("Error occurred during booking request cron job:", error);
        }
    });

};

