const models = require('../models');
const helper = require('../helpers/helper')
const { Validator } = require('node-input-validator');
const Sequelize = require("sequelize");
const { Op, fn, col } = Sequelize;
const moment = require('moment');
const currentDate = moment().format('YYYY-MM-DD');
const currentTime = moment().format('HH:mm:ss');
const stripe = require('stripe')('sk_test_51PvYN9AZ3KtiMZbRRnumfMRxTvGl3vq20RmJyEsj7HlLaOZJD0uYJaq94bQX2cWLWgbqEymTEYpTtPPyw1QxUp2M002EQfOIyC');
const { Transaction } = require('sequelize');

module.exports = {

    withdrawalRequest: async (req, res) => {
        try {
            let v = new Validator(req.body, {
                bank_id: 'required',
                requested_amount: 'required',
            });

            let errorsResponse = await helper.checkValidation(v)
            if (errorsResponse) {
                return helper.failed(res, errorsResponse)
            }

            const requestedAmount = parseFloat(req.body.requested_amount);
            if (isNaN(requestedAmount) || requestedAmount <= 0) {
                return helper.failed(res, "Requested amount should be a positive number greater than zero.");
            }

            const driver = await models.users.findOne({
                where: { id: req.user.id },
                attributes: ['wallet'],
                raw: true
            });

            if (!driver || parseFloat(driver.wallet) <= 0) {
                return helper.failed(res, "Your wallet is empty.");
            }

            if (requestedAmount > parseFloat(driver.wallet)) {
                return helper.failed(res, "Insufficient funds in your wallet.");
            }

            let withdrawal = {
                driver_id: req.user.id,
                bank_id: req.body.bank_id || 0,
                requested_amount: requestedAmount,
                driver_comment: req.body.driver_comment || "",
                date: currentDate,
                time: currentTime,
            }

            let createRequest = await models.withdrawals.create(withdrawal)
            if (createRequest) {
                // update left balance 
                let left = parseFloat(driver.wallet) - requestedAmount;
                await models.users.update({
                    wallet: left,
                }, {
                    where: {
                        id: req.user.id
                    }
                })

                let withdrawalData = await models.withdrawals.findOne({
                    include: [
                        {
                            model: models.banks
                        }
                    ],
                    where: {
                        id: createRequest.id
                    }
                })

                return helper.success(res, "Withdrawal request sent to admin successfully.", withdrawalData);
            } else {
                return helper.failed(res, "Something went wrong")
            }
        } catch (error) {
            console.error("Error in withdrawalRequest:", error);
            return helper.failed(res, error);
        }
    },

    allWithdrawalRequests: async (req, res) => {
        try {
            const driverId = req.user.id;
            // Fetch all withdrawal requests for the driver with bank information
            const withdrawalRequests = await models.withdrawals.findAll({
                include: [
                    {
                        model: models.banks,
                    }
                ],
                where:
                {
                    driver_id: driverId
                },
                order: [['id', 'DESC']]
            });

            let userData = await models.users.findOne({
                attributes: [
                    "id",
                    "email",
                    "country_code",
                    "phone_number",
                    "wallet"
                ],
                where: {
                    id: driverId
                },
                raw: true,
            });

            let response = {
                wallet: userData.wallet,
                withdrawalRequests
            }
            return helper.success(res, "Withdrawal requests retrieved successfully.", response);

        } catch (error) {
            console.error("Error in allWithdrawalRequests:", error);
            return helper.failed(res, error);
        }
    },

    tranferMoney: async (req, res) => {
        try {
            const t = await models.sequelize.transaction();

            let v = new Validator(req.body, {
                requested_amount: 'required',
            });

            let errorsResponse = await helper.checkValidation(v)
            if (errorsResponse) {
                return helper.failed(res, errorsResponse)
            }

            const requestedAmount = parseFloat(req.body.requested_amount);
            if (isNaN(requestedAmount) || requestedAmount <= 0) {
                return helper.failed(res, "Requested amount should be a positive number greater than zero.");
            }

            const driver = await models.users.findOne({
                where: { id: req.user.id },
                attributes: ['wallet', 'stripeAccountId'],
                raw: true
            });

            if (!driver || parseFloat(driver.wallet) <= 0) {
                return helper.failed(res, "Your wallet is empty.");
            }

            if (requestedAmount > parseFloat(driver.wallet)) {
                return helper.failed(res, "Insufficient funds in your wallet.");
            }

            if (!driver.stripeAccountId) {
                return helper.failed(res, "Stripe account not linked.");
            }

            let withdrawal = {
                driver_id: req.user.id,
                bank_id: req.body.bank_id || 0,
                requested_amount: requestedAmount,
                driver_comment: req.body.driver_comment || "",
                date: currentDate,
                time: currentTime,
            }

            let createRequest = await models.withdrawals.create(withdrawal, { transaction: t })
            if (createRequest) {
                // update left balance 

                transfer = await stripe.transfers.create({
                    amount: requestedAmount * 100,
                    currency: 'usd',
                    destination: driver.stripeAccountId,
                });

                if (transfer && transfer.id) {
                    await models.withdrawals.update({
                        fundTransferId: transfer.id
                    }, {
                        where: {
                            id: createRequest.id
                        }
                    })
                } else {
                    return helper.failed(res, "Failed to process the transfer with Stripe.");
                }

                let left = parseFloat(driver.wallet) - requestedAmount;
                await models.users.update({
                    wallet: left,
                }, {
                    where: {
                        id: req.user.id
                    }
                })

                let withdrawalData = await models.withdrawals.findOne({
                    include: [
                        {
                            model: models.banks
                        }
                    ],
                    where: {
                        id: createRequest.id
                    }
                })

                return helper.success(res, "Fund transfer successfully.", withdrawalData);
            } else {
                return helper.failed(res, "Something went wrong")
            }
        } catch (error) {
            console.error("Error in withdrawalRequest:", error);
            return helper.failed(res, error);
        }
    }

};
