const models = require('../models');
const helper = require('../helpers/helper')
const { Validator } = require('node-input-validator');
const Sequelize = require("sequelize");
const { Op, fn, col } = Sequelize;

module.exports = {
    addRating: (io) => {
        return async (req, res) => {
            try {
                // Validate the request input
                let validator = new Validator(req.body, {
                    driver_id: 'required',
                    booking_id: 'required',
                    no_of_rating: 'required',
                });

                let validationErrors = await helper.checkValidation(validator);
                if (validationErrors) {
                    return helper.failed(res, validationErrors);
                }

                // Create the rating
                let createRating = await models.ratings.create({
                    driver_id: req.body.driver_id,
                    user_id: req.user.id,
                    booking_id: req.body.booking_id,
                    no_of_rating: req.body.no_of_rating,
                    comments: req.body.comments || "",
                });

                if (createRating) {
                    // Fetch driver details
                    let driverDetails = await models.users.findOne({
                        where: { id: req.body.driver_id },
                        attributes: ['device_token', 'device_type', 'role'],
                        raw: true,
                    });

                    let message = `${req.user.first_name} ${req.user.last_name} gave you a rating.`;
                    let notificationData = {
                        sender_id: req.user.id,
                        receiver_id: req.body.driver_id,
                        user_type: driverDetails.role,
                        message: message,
                        type: 4, // Assuming type 4 represents user rating
                        booking_id: req.body.booking_id,
                        device_token: driverDetails.device_token,
                        device_type: driverDetails.device_type,
                    };

                    // Uncomment this line to send a push notification to the driver
                    if (notificationData.device_token) {
                        helper.sendPushToAndroid(notificationData);
                    }

                    // Save notification to the database
                    await models.notifications.create({
                        user_by: req.user.id,
                        user_to: req.body.driver_id,
                        status: 0,
                        comment: message,
                        type: notificationData.type,
                        is_read: 0,
                        booking_id: req.body.booking_id,
                    });

                    // Fetch the driver's socket information
                    let driverSocket = await models.socket_users.findOne({
                        where: { user_id: req.body.driver_id },
                        attributes: ['socket_id'],
                        raw: true,
                    });

                    // Fetch the rating details
                    let ratingDetails = await models.ratings.findOne({
                        include: [
                            { model: models.users },  // User who gave the rating
                            { model: models.bookings } // Booking details
                        ],
                        where: { id: createRating.id },
                    });

                    // Emit the rating details to the driver
                    if (driverSocket) {
                        io.to(driverSocket.socket_id).emit('userRating', ratingDetails);
                    }

                    return helper.success(res, "Rating sent successfully.", ratingDetails);
                } else {
                    return helper.failed(res, "Failed to create rating.");
                }

            } catch (error) {
                console.error("Error in addRating:", error);
                return helper.failed(res, "An error occurred while adding the rating.");
            }
        }
    },

    driverRating: async (req, res) => {
        try {
            let validator = new Validator(req.body, {
                driver_id: 'required',
            });

            let validationErrors = await helper.checkValidation(validator);
            if (validationErrors) {
                return helper.failed(res, validationErrors);
            }

            let driverDetails = await models.users.findOne({
                attributes: {
                    include: [
                        [
                            Sequelize.literal(
                                `(SELECT IFNULL(AVG(no_of_rating), 0) FROM ratings WHERE ratings.driver_id = ${req.body.driver_id})`
                            ),
                            'average_rating'
                        ],
                        [
                            Sequelize.literal(
                                `(SELECT COUNT(*) FROM ratings WHERE ratings.driver_id = ${req.body.driver_id})`
                            ),
                            'rating_count'
                        ]
                    ],
                },
                include: [
                    {
                        model: models.vehicle_types,
                        as: 'vehicleType'
                    },
                    {
                        model: models.vehicle_models,
                    }
                ],
                where: {
                    id: req.body.driver_id
                }
            });

            let ratingDetails = await models.ratings.findAll({
                include: [
                    { model: models.users },
                    { model: models.bookings }
                ],
                where: { driver_id: req.body.driver_id },
            });

            let response = {
                driverDetails,
                ratingDetails
            }

            return helper.success(res, "Rating list!", response);
        } catch (error) {
            console.error("Error in rating list:", error);
            return helper.failed(res, error);

        }
    }

};
