const models = require('../models');
const helper = require('../helpers/helper')
const { Validator } = require('node-input-validator');
const Sequelize = require("sequelize");
const { Op, fn, col } = Sequelize;
const cron = require('node-cron');
const stripe = require('stripe')('sk_test_51PvYN9AZ3KtiMZbRRnumfMRxTvGl3vq20RmJyEsj7HlLaOZJD0uYJaq94bQX2cWLWgbqEymTEYpTtPPyw1QxUp2M002EQfOIyC');

module.exports = {
    createBooking: (io) => {
        return async (req, res) => {

            try {
                let v = new Validator(req.body, {
                    vehicle_id: 'required',
                    destination_from: 'required',
                    destination_from_lat: 'required',
                    destination_from_long: 'required',
                    destination_to: 'required',
                    destination_to_lat: 'required',
                    destination_to_long: 'required',
                    price: 'required',
                });

                let errorsResponse = await helper.checkValidation(v)
                if (errorsResponse) {
                    return helper.failed(res, errorsResponse)
                }

                let bookingDat = {
                    user_id: req.user.id,
                    vehicle_id: req.body.vehicle_id ? req.body.vehicle_id : 0,
                    destination_from: req.body.destination_from ? req.body.destination_from : '',
                    destination_from_lat: req.body.destination_from_lat ? req.body.destination_from_lat : '',
                    destination_from_long: req.body.destination_from_long ? req.body.destination_from_long : '',
                    destination_to: req.body.destination_to ? req.body.destination_to : '',
                    destination_to_lat: req.body.destination_to_lat ? req.body.destination_to_lat : '',
                    destination_to_long: req.body.destination_to_long ? req.body.destination_to_long : '',
                    no_of_person: req.body.no_of_person ? req.body.no_of_person : 0,
                    is_schedule: req.body.is_schedule ? req.body.is_schedule : 0,
                    comment: req.body.comment ? req.body.comment : '',
                    booking_date: req.body.booking_date ? req.body.booking_date : '',
                    booking_time: req.body.booking_time ? req.body.booking_time : '',
                    price: req.body.price ? req.body.price : '',
                    distance: req.body.distance ? req.body.distance : ''
                }
                if (req.body.coupon_id) {
                    bookingDat.coupon_id = req.body.coupon_id
                }

                if (req.body.old_price) {
                    bookingDat.old_price = req.body.old_price
                }
                let createBooking = await models.bookings.create(bookingDat);

                createBooking = createBooking.toJSON();
                delete createBooking.createdAt;
                delete createBooking.updatedAt;

                let bookingData = await models.bookings.findOne({
                    include: [
                        {
                            model: models.vehicle_types,
                        }
                    ],
                    where: {
                        id: createBooking.id
                    }
                })

                // Find drivers

                const whereCondition = {
                    role: 1,
                    vehicle_type: bookingData.vehicle_id,
                    id: {
                        [Op.ne]: bookingData.user_id,
                    },
                };

                let sendPushToDriver = await models.users.findAll({
                    attributes: [
                        "id",
                        "first_name",
                        "last_name",
                        "email",
                        "country_code",
                        "phone_number",
                        "latitude",
                        "longitude",
                        "device_type",
                        "device_token",
                        [
                            Sequelize.literal(
                                `6371 * acos(cos(radians(${bookingData.destination_from_lat})) * cos(radians(latitude)) * cos(radians(${bookingData.destination_from_long}) - radians(longitude)) + sin(radians(${bookingData.destination_from_lat})) * sin(radians(latitude)))`
                            ),
                            "distance",
                        ],
                    ],
                    where: whereCondition,
                    having: {
                        distance: {
                            [Op.lte]: 500,
                        },
                    },
                    raw: true,
                    order: [["id", "desc"]],
                });
                console.log("=================", sendPushToDriver)
                // if (sendPushToDriver) {
                //     for (i = 0; i < sendPushToDriver.length; i++) {
                //         await models.request_send_to_drivers.create({
                //             booking_id: bookingData.id,
                //             driver_id: sendPushToDriver[i].id,
                //         })
                //         let message = `New booking from ${bookingData.destination_from} to ${bookingData.destination_to} `;
                //         let type = 1;
                //         let booking_id = bookingData.id;
                //         let device_token = sendPushToDriver[i].device_token;
                //         let device_type = sendPushToDriver[i].device_type;

                //         let data = {
                //             sender_id: req.user.id,
                //             receiver_id: sendPushToDriver[i].id,
                //             user_type: sendPushToDriver[i].role,
                //             message: message,
                //             type: type,
                //             booking_id: booking_id,
                //             device_token: device_token,
                //             device_type: device_type,
                //         };
                //         console.log("data", data);
                //         if (device_token) {
                //             helper.sendPushToAndroid(data);
                //         }

                //         let driverSockets = await models.socket_users.findOne({
                //             where: {
                //                 user_id: sendPushToDriver[i].id,
                //             },
                //             raw: true,
                //         });

                //         io.to(driverSockets && driverSockets.socket_id)
                //             .emit('bookingRequest', bookingData);
                //     }
                // }
                return helper.success(res, "Booking created successfully.", createBooking);

            } catch (error) {
                console.log("booking error", error);
                return helper.failed(res, error);
            }
        }
    },

    bookingDetails: async (req, res) => {
        try {
            let v = new Validator(req.body, {
                booking_id: 'required',
            });

            let errorsResponse = await helper.checkValidation(v)
            if (errorsResponse) {
                return helper.failed(res, errorsResponse)
            }

            let bookingDetails = await models.bookings.findOne({
                attributes: {
                    include: [
                        [
                            Sequelize.literal(
                                `(SELECT COUNT(*) FROM ratings WHERE ratings.booking_id = ${req.body.booking_id}) > 0`
                            ),
                            'is_rated'
                        ],
                    ],
                },
                include: [
                    {
                        model: models.vehicle_types,
                    },
                    {
                        model: models.users,
                    },
                    {
                        model: models.users,
                        as: "driver_details",
                        include: [
                            {
                                model: models.vehicle_models
                            }
                        ]
                    },
                    {
                        model: models.ratings
                    }
                ],
                where: {
                    id: req.body.booking_id
                }
            })

            return helper.success(res, "Booking details!", bookingDetails);
        } catch (error) {
            console.log("booking error", error);
            return helper.failed(res, error);
        }
    },

    sendBookingRequest: (io) => {
        return async (req, res) => {
            try {
                let v = new Validator(req.body, {
                    booking_id: 'required',
                });
                // console.log(req.socket); return
                // console.log(req.io);
                let errorsResponse = await helper.checkValidation(v)
                if (errorsResponse) {
                    return helper.failed(res, errorsResponse)
                }

                let bookingData = await models.bookings.findOne({
                    include: [
                        {
                            model: models.vehicle_types,
                        }
                    ],
                    where: {
                        id: req.body.booking_id
                    }
                })

                const whereCondition = {
                    role: 1,
                    vehicle_type: bookingData.vehicle_id,
                    id: {
                        [Op.ne]: bookingData.user_id,
                    },
                };

                let sendPushToDriver = await models.users.findAll({
                    attributes: [
                        "id",
                        "first_name",
                        "last_name",
                        "email",
                        "country_code",
                        "phone_number",
                        "latitude",
                        "longitude",
                        "device_type",
                        "device_token",
                        [
                            Sequelize.literal(
                                `6371 * acos(cos(radians(${bookingData.destination_from_lat})) * cos(radians(latitude)) * cos(radians(${bookingData.destination_from_long}) - radians(longitude)) + sin(radians(${bookingData.destination_from_lat})) * sin(radians(latitude)))`
                            ),
                            "distance",
                        ],
                    ],
                    where: whereCondition,
                    having: {
                        distance: {
                            [Op.lte]: 25,
                        },
                    },
                    raw: true,
                    order: [["id", "desc"]],
                });

                // console.log("sendPushToDriver", sendPushToDriver); return

                if (sendPushToDriver) {
                    for (i = 0; i < sendPushToDriver.length; i++) {
                        await models.request_send_to_drivers.create({
                            booking_id: bookingData.id,
                            driver_id: sendPushToDriver[i].id,
                        })
                        let message = `New booking from ${bookingData.destination_from} to ${bookingData.destination_to} `;
                        let type = 1;
                        let booking_id = bookingData.id;
                        let device_token = sendPushToDriver[i].device_token;
                        let device_type = sendPushToDriver[i].device_type;

                        let data = {
                            sender_id: bookingData.id,
                            receiver_id: sendPushToDriver[i].id,
                            user_type: sendPushToDriver[i].role,
                            message: message,
                            type: type,
                            booking_id: booking_id,
                            device_token: device_token,
                            device_type: device_type,
                        };
                        console.log("data", data);
                        helper.sendPushToAndroid(data);

                        let driverSockets = await models.socket_users.findOne({
                            where: {
                                user_id: sendPushToDriver[i].id,
                            },
                            raw: true,
                        });

                        io.to(driverSockets && driverSockets.socket_id)
                            .emit('bookingRequest', bookingData);
                    }
                }

                return helper.success(res, "Booking request sent successfully!", bookingData);
            } catch (error) {
                console.log("booking error", error);
                return helper.failed(res, error);
            }
        }
    },

    driverBookingRequest: async (req, res) => {
        try {
            let bookingDetails = await models.request_send_to_drivers.findAll({
                include: [
                    {
                        model: models.bookings,
                    }
                ],
                where: {
                    driver_id: req.user.id
                }
            })

            return helper.success(res, "Booking list!", bookingDetails);
        } catch (error) {
            console.log("booking error", error);
            return helper.failed(res, error);
        }
    },

    myBookings: async (req, res) => {
        try {
            let v = new Validator(req.body, {
                type: 'required',
            });

            let errorsResponse = await helper.checkValidation(v)
            if (errorsResponse) {
                return helper.failed(res, errorsResponse)
            }
            let where = {}
            where.user_id = req.user.id

            // Handle different status types
            if (req.body.type !== undefined && req.body.type !== null) {
                switch (req.body.type) {
                    case '0': // Pending
                        where.status = 0;
                        where.is_schedule = 0;
                        break;
                    case '1': // Accepted
                        // where.status = 1;
                        where.status = {
                            [Op.in]: [1, 2, 5]
                        };
                        where.booking_date = {
                            [Op.lte]: new Date()
                        };
                        break;
                    case '2': // Ongoing
                        // where.status = 2;
                        where.status = {
                            [Op.in]: [2, 5]
                        };
                        break;
                    case '3': // Cancelled
                        where.status = 3;
                        break;
                    case '4': // Completed
                        // where.status = 4;
                        where.status = {
                            [Op.in]: [4, 3]
                        };
                        break;
                    case '5': // Scheduled
                        where.is_schedule = 1;
                        where.status = {
                            [Op.in]: [0, 1]
                        };
                        where.booking_date = {
                            [Op.lte]: new Date(), // Current date must be earlier than the booking_date
                        };
                        break;
                    default:
                        // If an invalid type is provided, return all bookings
                        break;
                }
            }

            let bookingList = await models.bookings.findAll({
                include: [
                    {
                        model: models.vehicle_types,
                    },
                    {
                        model: models.users,
                    },
                    {
                        model: models.users,
                        as: "driver_details"
                    }
                ],
                where: where,
                order: [['id', 'desc']]
            })

            return helper.success(res, "Booking list!", bookingList);
        } catch (error) {
            console.log("booking error", error);
            return helper.failed(res, error);
        }
    },

    driverBookings: async (req, res) => {
        try {
            let v = new Validator(req.body, {
                type: 'required',
            });

            let errorsResponse = await helper.checkValidation(v)
            if (errorsResponse) {
                return helper.failed(res, errorsResponse)
            }
            let where = {}
            where.driver_id = req.user.id

            if (req.body.type !== undefined && req.body.type !== null) {
                switch (req.body.type) {
                    case '0': // Pending
                        where.status = 0;
                        where.is_schedule = 0;
                        break;
                    case '1': // Accepted
                        // where.status = 1;
                        where.status = {
                            [Op.in]: [1, 2, 5]
                        };
                        where.booking_date = {
                            [Op.lte]: new Date()
                        };
                        break;
                    case '2': // Ongoing
                        // where.status = 2;
                        where.status = {
                            [Op.in]: [2, 5]
                        };
                        break;
                    case '3': // Cancelled
                        where.status = 3;
                        break;
                    case '4': // Completed
                        // where.status = 4;
                        where.status = {
                            [Op.in]: [4, 3]
                        };
                        break;
                    case '5': // Scheduled
                        where.is_schedule = 1;
                        where.status = {
                            [Op.in]: [0, 1]
                        };
                        where.booking_date = {
                            [Op.lte]: new Date(), // Current date must be earlier than the booking_date
                        };
                        break;
                    default:
                        // If an invalid type is provided, return all bookings
                        break;
                }
            }
            let bookingList = await models.bookings.findAll({
                include: [
                    {
                        model: models.vehicle_types,
                    },
                    {
                        model: models.users,
                    },
                    {
                        model: models.users,
                        as: "driver_details"
                    }
                ],
                where: where,
                order: [['id', 'desc']]
            })

            return helper.success(res, "Booking list!", bookingList);
        } catch (error) {
            console.log("booking error", error);
            return helper.failed(res, error);
        }
    },

    driverAcceptRejectBooking: (io) => {
        return async (req, res) => {
            try {
                // Validate request input
                let validator = new Validator(req.body, {
                    booking_id: 'required',
                    status: 'required'
                });
                let validationErrors = await helper.checkValidation(validator);
                if (validationErrors) {
                    return helper.failed(res, validationErrors);
                }

                if (req.body.status == 1) {
                    // Update the booking with the accepted driver
                    await models.bookings.update({
                        driver_id: req.user.id,
                        status: 1
                    }, {
                        where: { id: req.body.booking_id }
                    });

                    // Remove all other pending requests for this booking
                    await models.request_send_to_drivers.destroy({
                        where: {
                            booking_id: req.body.booking_id,
                        }
                    });

                    let bookingDetails = await models.bookings.findOne({
                        include: [
                            { model: models.users },
                            { model: models.users, as: "driver_details" },
                        ],
                        where: { id: req.body.booking_id },
                    });

                    // Prepare notification message and data
                    let message = `${bookingDetails.driver_details.first_name} ${bookingDetails.driver_details.last_name} has accepted the ride request.`;
                    let notificationData = {
                        sender_id: req.user.id,
                        receiver_id: bookingDetails.user.id,
                        user_type: bookingDetails.user.role,
                        message: message,
                        type: 3,  // Assuming type 2 is for booking requests
                        booking_id: req.body.booking_id,
                        device_token: bookingDetails.user.device_token,
                        device_type: bookingDetails.user.device_type,
                    };

                    // Send push notification if the user has a device token
                    if (notificationData.device_token) {
                        helper.sendPushToAndroid(notificationData);
                    }

                    // Save notification to the database
                    await models.notifications.create({
                        user_by: req.user.id,
                        user_to: bookingDetails.user.id,
                        status: 0,
                        comment: message,
                        type: notificationData.type,
                        is_read: 0,
                        booking_id: req.body.booking_id,
                    });

                    // Retrieve user's socket information and emit booking update
                    let userSocket = await models.socket_users.findOne({
                        where: { user_id: bookingDetails.user_id },
                        raw: true,
                    });

                    if (userSocket) {
                        io.to(userSocket.socket_id).emit('bookingRequest', bookingDetails);
                    }
                    return helper.success(res, "Ride request accepted successfully!", bookingDetails);

                } else {
                    // If the driver rejects the ride, remove only their request
                    // await models.request_send_to_drivers.destroy({
                    //     where: {
                    //         booking_id: req.body.booking_id,
                    //         driver_id: req.user.id,
                    //     }
                    // });

                    await models.request_send_to_drivers.update(
                        { status: 2 }, // Set the status to 2
                        {
                            where: {
                                booking_id: req.body.booking_id,
                                driver_id: req.user.id,
                            }
                        }
                    );


                    return helper.success(res, "Ride request cancelled successfully!", {});
                }
            } catch (error) {
                console.error("Error handling booking request:", error);
                return helper.failed(res, error);
            }
        };
    },

    userAcceptRejectDriver: (io) => {
        return async (req, res) => {
            try {
                let validator = new Validator(req.body, {
                    request_id: 'required',
                    booking_id: 'required',
                });

                let validationErrors = await helper.checkValidation(validator);
                if (validationErrors) {
                    return helper.failed(res, validationErrors);
                }

                let existingRequest = await models.request_send_to_drivers.findOne({
                    where: { id: req.body.request_id }
                });

                if (existingRequest) {
                    await existingRequest.update({
                        status: 1
                    });

                    await models.bookings.update({
                        driver_id: existingRequest.driver_id
                    }, {
                        where: { id: req.body.booking_id }
                    });

                    let bookingDetails = await models.bookings.findOne({
                        include: [
                            { model: models.users },
                            { model: models.users, as: "driver_details" },
                        ],
                        where: { id: req.body.booking_id },
                    });


                    let message = `${bookingDetails.user.first_name} ${bookingDetails.user.last_name} accepted your ride request.`;

                    // Send notification to the driver
                    let notificationData = {
                        sender_id: req.user.id,
                        receiver_id: existingRequest.driver_id,
                        user_type: bookingDetails.driver_details.role,
                        message: message,
                        type: 3,
                        booking_id: req.body.booking_id,
                        device_token: bookingDetails.driver_details.device_token,
                        device_type: bookingDetails.driver_details.device_type,
                    };

                    if (notificationData.device_token) {
                        // helper.sendPushToAndroid(notificationData);
                    }

                    // Save notification to the database
                    await models.notifications.create({
                        user_by: bookingDetails.user.id,
                        user_to: existingRequest.driver_id,
                        status: 0,
                        comment: message,
                        type: notificationData.type,
                        is_read: 0,
                        booking_id: req.body.booking_id,
                    });

                    // Retrieve driver's socket information and emit acceptance event
                    let driverSocket = await models.socket_users.findOne({
                        where: { user_id: existingRequest.driver_id },
                        raw: true,
                    });

                    if (driverSocket) {
                        io.to(driverSocket.socket_id).emit('driverAccepted', bookingDetails);
                    }

                    return helper.success(res, "Driver accepted successfully!", bookingDetails);
                } else {
                    return helper.failed(res, 'Invalid request. The specified request does not exist.');
                }
            } catch (error) {
                console.error("Error handling driver acceptance:", error);
                return helper.failed(res, "An error occurred while processing the driver acceptance.");
            }
        };
    },

    driverNewBookingList: async (req, res) => {
        try {
            let bookings = await models.request_send_to_drivers.findAll({
                include: [
                    {
                        model: models.bookings,
                        include: [
                            {
                                model: models.users
                            }
                        ],
                    }
                ],
                where: {
                    driver_id: req.user.id,
                    status: 0
                },
                order: [['id', 'desc']]
            });

            return helper.success(res, "New booking list!", bookings);

        } catch (error) {
            console.error("Error handling driver acceptance:", error);
            return helper.failed(res, "An error occurred while processing the driver acceptance.");
        }
    },

    bookingCancel: (io) => {
        return async (req, res) => {
            try {
                let validator = new Validator(req.body, {
                    booking_id: 'required',
                    comment: 'required',
                    cancel_amount: 'required'
                });

                let validationErrors = await helper.checkValidation(validator);
                if (validationErrors) {
                    return helper.failed(res, validationErrors);
                }

                let bookingData = await models.bookings.findOne({
                    where: { id: req.body.booking_id },
                    raw: true
                });

                if (req.body.cancel_amount > 0) {
                    if (bookingData.transaction_id && bookingData.stripe_status == 'succeeded') {
                        const amountInCents = Math.round(req.body.cancel_amount * 100);

                        const refund = await stripe.refunds.create({
                            payment_intent: bookingData.transaction_id,
                            amount: amountInCents,
                        });

                        console.log("refund", refund);
                        await models.bookings.update({
                            cancel_data: refund,
                            cancel_amount: req.body.cancel_amount || 0,
                        }, {
                            where: { id: req.body.booking_id }
                        });
                    }

                }
                // Update booking status and set the user who cancelled the booking
                await models.bookings.update({
                    status: 3, // 3 indicates cancelled
                    cancel_by: req.user.id,
                    comment: req.body.comment,
                }, {
                    where: { id: req.body.booking_id }
                });

                // Fetch booking details including user, driver, and cancel information
                let bookingDetails = await models.bookings.findOne({
                    include: [
                        { model: models.users },
                        { model: models.users, as: "driver_details" },
                        { model: models.users, as: "cancelBy" },
                    ],
                    where: { id: req.body.booking_id },
                });

                // Determine the role of the user who cancelled the ride
                let message, receiverId, receiverToken, receiverDeviceType;
                const cancelById = bookingDetails.cancel_by;

                if (cancelById === bookingDetails.user.id) {
                    // User cancelled the ride, notify the driver
                    message = `${bookingDetails.cancelBy.first_name} ${bookingDetails.cancelBy.last_name} has cancelled the ride.`;
                    receiverId = bookingDetails.driver_details.id;
                    receiverToken = bookingDetails.driver_details.device_token;
                    receiverDeviceType = bookingDetails.driver_details.device_type;
                } else if (cancelById === bookingDetails.driver_details.id) {
                    // Driver cancelled the ride, notify the user
                    message = `Your driver ${bookingDetails.driver_details.first_name} ${bookingDetails.driver_details.last_name} has cancelled the ride.`;
                    receiverId = bookingDetails.user.id;
                    receiverToken = bookingDetails.user.device_token;
                    receiverDeviceType = bookingDetails.user.device_type;
                }

                // Send push notification if the receiver has a device token
                if (receiverToken) {
                    let notificationData = {
                        sender_id: cancelById,
                        receiver_id: receiverId,
                        message: message,
                        type: 2,
                        booking_id: req.body.booking_id,
                        device_token: receiverToken,
                        device_type: receiverDeviceType,
                    };
                    helper.sendPushToAndroid(notificationData);

                }

                // Save notification to the database
                await models.notifications.create({
                    user_by: cancelById,
                    user_to: receiverId,
                    status: 0,
                    comment: message,
                    type: 2,
                    is_read: 0,
                    booking_id: req.body.booking_id,
                });

                // Retrieve receiver's socket information and emit cancellation event
                let receiverSocket = await models.socket_users.findOne({
                    where: { user_id: receiverId },
                    raw: true,
                });

                if (receiverSocket) {
                    io.to(receiverSocket.socket_id).emit('rideCancelled', bookingDetails);
                }

                return helper.success(res, "Ride cancelled successfully!", bookingDetails);
            } catch (error) {
                console.error("Error handling ride cancellation:", error);
                return helper.failed(res, "An error occurred while processing the ride cancellation.");
            }
        };
    },

    deleteExpiredBookings: async (req, res) => {
        try {
            // Fetch expired bookings with status 0
            let expiredBookings = await models.bookings.findAll({
                where: {
                    booking_date: {
                        [Op.lt]: new Date() // Find bookings where booking_date is earlier than today
                    },
                    status: 0
                },
                attributes: ['id']
            });

            if (expiredBookings.length > 0) {
                // Extract booking IDs
                const bookingIds = expiredBookings.map(booking => booking.id);

                // Delete related records from request_send_to_drivers table
                await models.request_send_to_drivers.destroy({
                    where: {
                        booking_id: {
                            [Op.in]: bookingIds
                        }
                    }
                });

                // Delete expired bookings
                await models.bookings.destroy({
                    where: {
                        id: {
                            [Op.in]: bookingIds
                        }
                    }
                });

                return helper.success(res, "Expired bookings and related records deleted successfully.");
            } else {
                return helper.success(res, "No expired bookings found with status 0.");
            }
        } catch (error) {
            console.error("Error deleting expired bookings:", error);
            return helper.failed(res, "An error occurred while deleting expired bookings.");
        }
    }

};
